/*
 * Turtle3D.java
 *
 * Created on 02 September 2004, 17:20
 *
 * This class is responsible for keeping the position and orientation
 * of the turtle in 3 dimensions. The class also implements methods
 * for the direction change and the rotation required to allign 
 * an object with the turtle's heading vector.
 *
 * @author  Alexander Dimitropoulos
 * @version 2
 */
import javax.vecmath.Vector3d;
import javax.vecmath.AxisAngle4d;

public class Turtle3D {
    
    private double x;       //the x coordinate
    private double y;       //the y coordinate
    private double z;       //the z coordinate
    private double alpha;   //the direction angle (DOL-Systems)
    private float length;   //the length of the section
    private float width;    //the width of the section
    private float tempWidth;    //temporal width, used in parametric strings
    private Vector3d H;     //the heading vector
    private Vector3d L;     //the vector to the left
    private Vector3d U;     //the vector to the up direction
    private double[] directionVec;    //vector array
    
    /** Creates a new instance of Turtle3D */
    public Turtle3D() {
        directionVec = new double[3];
    }
    
    /**
     * Creates a new instance of Turtle3D
     */
    public Turtle3D(double x, double y, double z) {
        this.x = x;
        this.y = y;
        this.z = z;
        directionVec = new double[3];
    }
    
    /**
     * Creates a new instance of Turtle3D
     */
    public Turtle3D(double x, double y, double z, double alpha, float length, float width) {
        this.x = x;
        this.y = y;
        this.z = z;
        this.alpha = alpha;
        this.length = length;
        this.width = width;
        directionVec = new double[3];
    }
    
    /**
     * Set of mutator methods
     * used to change turtle values
     *
     * @param x     the x-coordinate of the turtle
     */
    public void setX(double x) {
        this.x = x;
    }
    
    /**
     * @param y     the y-coordinate of the turtle
     */
    public void setY(double y) {
        this.y = y;
    }
    
    /**
     * @param z     the z-coordinate of the turtle
     */
    public void setZ(double z) {
        this.z = z;
    }
    
    /**
     * @param x     the x-coordinate of the turtle
     * @param y     the y-coordinate of the turtle
     * @param z     the z-coordinate of the turtle
     */
    public void setXYZ(double x, double y, double z) {
        this.x = x;
        this.y = y;
        this.z = z;
    }
    
    /**
     * @param alpha     the direction angle of the turtle
     */
    public void setAlpha(double alpha) {
        this.alpha = alpha;
    }
    
    /**
     * @param length    the length of the section to draw
     */
    public void setLength(float length) {
        this.length = length;
    }
    
    /**
     * @param width     the width of the section to draw
     */
    public void setWidth(float width) {
        this.width = width;
    }
    
    /**
     * @param tempWidth     the temporal width used in parametric strings
     *                      to create sections that have a smoothly
     *                      decreasing width
     */
    public void setTempWidth(float tempWidth) {
        this.tempWidth = tempWidth;
    }
    
    /**
     * @param H     the heading vector
     */
    public void setH(Vector3d H) {
        this.H = H;
    }
    
    /**
     * @param L     the left vector
     */
    public void setL(Vector3d L) {
        this.L = L;
    }
    
    /**
     * @param U     the up vector
     */
    public void setU(Vector3d U) {
        this.U = U;
    }
    
    /**
     * @param H     the heading vector
     * @param L     the left vector
     * @param U     the up vector
     */
    public void setHLU(Vector3d H, Vector3d L, Vector3d U) {
        this.H = H;
        this.L = L;
        this.U = U;
    }
    
    /**
     * @param decLen    the amount to decrease the length
     */
    public void decreaseLength(double decLen) {
        length *= decLen;
    }
    
    /**
     * @param decLen    the amount to decrease the width
     */
    public void decreaseWidth(double decWidth) {
        width *= decWidth;
    }
    
    
    /**
     *Set of accessor methods
     *Used to get turtle values
     */
    public double getX() {
        return x;
    }
    
    public double getY() {
        return y;
    }
    
    public double getZ() {
        return z;
    }
    
    public double getAlpha() {
        return alpha;
    }
    
    public float getLength() {
        return length;
    }
    
    public float getWidth() {
        return width;
    }
    
    public float getTempWidth() {
        return tempWidth;
    }
    
    public Vector3d getH() {
        return H;
    }
    
    public Vector3d getL() {
        return L;
    }
    
    public Vector3d getU() {
        return U;
    }
    
    public double getHeadingX() {
        H.get(directionVec);
        return directionVec[0];
    }
    
    public double getHeadingY() {
        H.get(directionVec);
        return directionVec[1];
    }
    
    public double getHeadingZ() {
        H.get(directionVec);
        return directionVec[2];
    }
    
    public double getLeftX() {
        L.get(directionVec);
        return directionVec[0];
    }
    
    public double getLeftY() {
        L.get(directionVec);
        return directionVec[1];
    }
    
    public double getLeftZ() {
        L.get(directionVec);
        return directionVec[2];
    }
    
    public double getUpX() {
        U.get(directionVec);
        return directionVec[0];
    }
    
    public double getUpY() {
        U.get(directionVec);
        return directionVec[1];
    }
      
    public double getUpZ() {
        U.get(directionVec);
        return directionVec[2];
    }
    
        
    /**
     * Set of mutator methods that change the orientation of the turtle
     *
     * method to move the turtle position to the next position
     * The method calculates the next position
     * by myltiplying the unit vector by the length of the segment
     */
    public void setNewPosition() {
        setX(x + length * getHeadingX());
        setY(y + length * getHeadingY());
        setZ(z + length * getHeadingZ());
    }
    
    /**
     * method to rotate the turtle heading to the left
     */
    public void turnLeftU() {
        double Hx, Hy, Hz;
        double Lx, Ly, Lz;
        double Ux, Uy, Uz;
        double a = alpha;
        
        Hx = getHeadingX() * Math.cos(a) - getLeftX() * Math.sin(a);
        Hy = getHeadingY() * Math.cos(a) - getLeftY() * Math.sin(a);
        Hz = getHeadingZ() * Math.cos(a) - getLeftZ() * Math.sin(a);
        
        Lx = getHeadingX() * Math.sin(a) + getLeftX() * Math.cos(a);
        Ly = getHeadingY() * Math.sin(a) + getLeftY() * Math.cos(a);
        Lz = getHeadingZ() * Math.sin(a) + getLeftZ() * Math.cos(a);
        
        setH(new Vector3d(Hx, Hy, Hz));
        setL(new Vector3d(Lx, Ly, Lz));
    }
    
    /**
     * this is the same as the turnLeftU method by the
     * heading turns right
     */
    public void turnRightU() {
        double Hx, Hy, Hz;
        double Lx, Ly, Lz;
        double Ux, Uy, Uz;
        double a = alpha;
        
        //same as before but the sign is vhanged to make the opposite angle
        //sin(-a)=-sin(a)
        Hx = getHeadingX() * Math.cos(a) + getLeftX() * Math.sin(a);
        Hy = getHeadingY() * Math.cos(a) + getLeftY() * Math.sin(a);
        Hz = getHeadingZ() * Math.cos(a) + getLeftZ() * Math.sin(a);
        
        Lx = -getHeadingX() * Math.sin(a) + getLeftX() * Math.cos(a);
        Ly = -getHeadingY() * Math.sin(a) + getLeftY() * Math.cos(a);
        Lz = -getHeadingZ() * Math.sin(a) + getLeftZ() * Math.cos(a);
        
        setH(new Vector3d(Hx, Hy, Hz));
        setL(new Vector3d(Lx, Ly, Lz));
    }
    
    /**
     * method to pitch the heading of the turtle down
     */
    public void pitchDownL() {
        double Hx, Hy, Hz;
        double Lx, Ly, Lz;
        double Ux, Uy, Uz;
        double a = alpha;
        
        
        Hx = getHeadingX() * Math.cos(a) + getUpX() * Math.sin(a);
        Hy = getHeadingY() * Math.cos(a) + getUpY() * Math.sin(a);
        Hz = getHeadingZ() * Math.cos(a) + getUpZ() * Math.sin(a);
        
        Ux = -getHeadingX() * Math.sin(a) + getUpX() * Math.cos(a);
        Uy = -getHeadingY() * Math.sin(a) + getUpY() * Math.cos(a);
        Uz = -getHeadingZ() * Math.sin(a) + getUpZ() * Math.cos(a);
        
        setH(new Vector3d(Hx, Hy, Hz));
        setU(new Vector3d(Ux, Uy, Uz));
    }
    
    /**
     * the same as for pitchDownL but the head is turn upwards
     */
    public void pitchUpL() {
        double Hx, Hy, Hz;
        double Lx, Ly, Lz;
        double Ux, Uy, Uz;
        double a = alpha;
         
        Hx = getHeadingX() * Math.cos(a) - getUpX() * Math.sin(a);
        Hy = getHeadingY() * Math.cos(a) - getUpY() * Math.sin(a);
        Hz = getHeadingZ() * Math.cos(a) - getUpZ() * Math.sin(a);
        
        Ux = getHeadingX() * Math.sin(a) + getUpX() * Math.cos(a);
        Uy = getHeadingY() * Math.sin(a) + getUpY() * Math.cos(a);
        Uz = getHeadingZ() * Math.sin(a) + getUpZ() * Math.cos(a);
              
        setH(new Vector3d(Hx, Hy, Hz));
        setU(new Vector3d(Ux, Uy, Uz));
    }
    
    /**
     * method to roll the turtle to the left
     */
    public void rollLeftH() {
        double Hx, Hy, Hz;
        double Lx, Ly, Lz;
        double Ux, Uy, Uz;
        double a = alpha;
        
        Lx = getLeftX() * Math.cos(a) + getUpX() * Math.sin(a);
        Ly = getLeftY() * Math.cos(a) + getUpY() * Math.sin(a);
        Lz = getLeftZ() * Math.cos(a) + getUpZ() * Math.sin(a);
        
        Ux = -getLeftX() * Math.sin(a) + getUpX() * Math.cos(a);
        Uy = -getLeftY() * Math.sin(a) + getUpY() * Math.cos(a);
        Uz = -getLeftZ() * Math.sin(a) + getUpZ() * Math.cos(a);
        
        setL(new Vector3d(Lx, Ly, Lz));
        setU(new Vector3d(Ux, Uy, Uz));
    }
    
    /**
     * the same as rollRigthH but the turtle turns to the right
     */
    public void rollRightH() {
        double Hx, Hy, Hz;
        double Lx, Ly, Lz;
        double Ux, Uy, Uz;
        double a = alpha;
        
        Lx = getLeftX() * Math.cos(a) - getUpX() * Math.sin(a);
        Ly = getLeftY() * Math.cos(a) - getUpY() * Math.sin(a);
        Lz = getLeftZ() * Math.cos(a) - getUpZ() * Math.sin(a);
        
        Ux = getLeftX() * Math.sin(a) + getUpX() * Math.cos(a);
        Uy = getLeftY() * Math.sin(a) + getUpY() * Math.cos(a);
        Uz = getLeftZ() * Math.sin(a) + getUpZ() * Math.cos(a);
        
        setL(new Vector3d(Lx, Ly, Lz));
        setU(new Vector3d(Ux, Uy, Uz));
    }
    
    /**
     * method to turn the turtle around the u-axis by 180
     */
    public void turnAroundU() {
        double Hx, Hy, Hz;
        double Lx, Ly, Lz;
        double Ux, Uy, Uz;
        double a = Math.PI;
        
        Hx = getHeadingX() * Math.cos(a) + getUpX() * Math.sin(a);
        Hy = getHeadingY() * Math.cos(a) + getUpY() * Math.sin(a);
        Hz = getHeadingZ() * Math.cos(a) + getUpZ() * Math.sin(a);
        
        Ux = -getHeadingX() * Math.sin(a) + getUpX() * Math.cos(a);
        Uy = -getHeadingY() * Math.sin(a) + getUpY() * Math.cos(a);
        Uz = -getHeadingZ() * Math.sin(a) + getUpZ() * Math.cos(a);
                
        setH(new Vector3d(Hx, Hy, Hz));
        setU(new Vector3d(Ux, Uy, Uz)); 
    }
    
    /**
     * method to move the turtle heading to the horizontal position
     */
    public void rotateToVertical() {
        L.cross(new Vector3d(0,1,0), H);
        L.normalize();
        U.cross(H, L);
    }
    
    
    /**
     * method to create the rotation of the object given 
     * a specific axis-angle
     * The axis and the angle of rotation are derived from the cross product
     * of the current vector heading and the required direction vector
     */
    public AxisAngle4d createRotationAxisAngle() {
        
        //set vector a pointing upwards (U=1, H=0, L=0)
        //this vector represent the current section orientation
        Vector3d a = new Vector3d(0,1,0);   
        //assign b to current direction vector
        Vector3d b = new Vector3d(getHeadingX(),getHeadingY(),getHeadingZ());
        
        AxisAngle4d axisangle;
        
        //check if direction vector is equal to original vector
        if (b.equals(a))
             axisangle = new AxisAngle4d(0.0d,1.0d,0.0d,0.0d);
        
        //check if direction vector is onsame axis but opposite directon
        //with original vector
        else if (b.equals(new Vector3d(0,-1,0)))
            axisangle = new AxisAngle4d(0,0,1,Math.PI);
        
        else {
            
            //vector n is the axis of rotation
            Vector3d n = new Vector3d();
            
            //get vector n with the cross product of a and b (axb)
            n.cross(a,b);
            //vector n = (length of n) x (unit vector of n)
            //length of n = Sin(theta)
            
            double theta = a.angle(b);
            
            //normalise the vector n
            n.normalize();
            
            axisangle = new AxisAngle4d(n,theta);
        }
        
        return axisangle;
    }
}
