/*
 * TreeString.java
 *
 * Created on 03 August 2004, 00:42 
 * 
 * This class implements string production methods. String production rules 
 * and parameters are stored into separate HashMaps
 * 
 * @author  Alexander Dimitropoulos
 * @version 2
 */

import java.util.HashMap;      // used for the production rules


public class TreeString {
    
    private String axiom;      //the axiom of the system
    private String generation; //the generation string
    private HashMap pRules;    //the production rule map
    private HashMap parameters;//the parameters map
    private int n;             //the number of derivations
    
    /** 
     * Creates a new instance of TreeString
     */
    public TreeString() {
        axiom = new String();
        pRules = new HashMap();
        parameters = new HashMap();
    }
    
    /**
     * ,ethod to get the derivations number from the user
     *
     * @param derivations   the number of derivations
     */
    public void setDerivations(int derivations) {
       n = derivations;
    }
    
    /**
     * ,ethod to get the axim of the system
     *
     * @param omega     the axiom of the L-System
     */
    public void setAxiom(String omega) {
       axiom = omega;         
    }
    
    /**
     * method to set the production rules
     *
     * @param rule      a production rule of the L-system
     */
    public void addRule(String rule) {
        
        String[] productrule = rule.split("->");
        String Key = productrule[0].trim();     //removing unwanted spaces
        String Value = productrule[1].trim();
        if (Key.indexOf('(')!=-1)
            Key = Key.substring(0,Key.indexOf('('));
            
        pRules.put(Key,Value);
     }    
    
    /**
     * Method to set the parameters of a parametric string
     *
     * @param param     a parameter value of a parametric system
     */
    public void addParameter(String param) {
        
        String[] params = param.split("=");
        String Key = params[0].trim();
        String Value = params[1].trim();        
        parameters.put(Key,Value);
    }
    
    /**
     * method to generate the production string
     */
    public String generateString() {
        
        generation = axiom;
        char tempChar;
        
        for (int j=0; j<n; j++) {
            
            char productionArray[] = generation.toCharArray();
            String tempgen = new String();
            
            for (int i = 0; i<productionArray.length; i++) {
                tempChar = productionArray[i];
                if (pRules.containsKey(String.valueOf(tempChar)))
                    tempgen += pRules.get(String.valueOf(tempChar));
                else
                    tempgen += productionArray[i];
            }
            
            generation = tempgen;
        }
        
        return generation;
    }
    
    /**
     * Method to generate a parametric sting
     */
    public String generateParametricString() {
        
        generation = axiom;
        
        /**
         * adding a space character at the end of the string 
         * to prevent an out of bounds error when checking 
         * if next character is ( in second if loop
         */
        generation += " ";
        
        char tempChar;
        String temp = new String();
        String params = new String();
        String param = new String();
        String L = new String("l");     //the length parameter
        String W = new String("w");     //the width parameter
        
        //repeat for n times where n is the number of derivations
        for (int j=0; j<n; j++) {
            
            char pArray[] = generation.toCharArray();
            String tempgen = new String();
            
            //repeat for the length of the character array
            for (int i = 0; i<pArray.length; i++) {
                tempChar = pArray[i];
                
                //check if current character matches a production rule
                if (pRules.containsKey(String.valueOf(tempChar))) {
                    temp = (String) pRules.get(String.valueOf(tempChar));
                    
                    //check if the next character is and opening brracket
                    if ( pArray[i+1]=='(') {
                        
                        params = generation.substring(generation.indexOf('(', i) , generation.indexOf(')',i)+1);
                        i = generation.indexOf(')',i); 
                        
                        //check if two parameters are contained in the brackets separated by ','
                        if (params.indexOf(',')!=-1) {
                            
                            String[] tempParams = getParameters(params);
                            
                            // remove key l,w
                            // add keys l & w
                            parameters.remove("l");
                            parameters.remove("w");
                            parameters.put(L,tempParams[0]);
                            parameters.put(W,tempParams[1]);

                            tempgen += replaceValuesInParenthesis(temp);
                        }
                        
                        //if only one parameter is inside the brackets
                        else {
                            
                            param = params.substring(params.indexOf('(')+1, params.indexOf(')'));
                            
                            // remove key l,w
                            // add new key with values l = w = param
                            parameters.remove("l");
                            parameters.remove("w");
                            parameters.put(L,param);
                            parameters.put(W,param);
    
                            tempgen += replaceValuesInParenthesis(temp);
                            
                        } //end one or two parameters in brackets check
                    } //end next character is an opening bracket check
                    
                    /**
                     * this part of the loop is used for parametric strings 
                     * with tropism. in this case, the axiom is given in
                     * the form !(1)F(200)>(45)A, where A is a production
                     * rule of the form A->!(vr)F(50)[&(a)F(50)A]
                     * and replacement of the values in brackets is required
                     */
                    else {
                        
                        tempgen += replaceValuesInParenthesis(temp);
                        
                    }
                    
                } //end if character is in production rules
                
                //if the character is not in production rules but is a bracket
                //replace the bracket's content with appropiate values
                else if (pArray[i] == '(') {
                    
                    tempgen += "(";
                    param = generation.substring(generation.indexOf('(',i)+1,generation.indexOf(')',i));
                    tempgen += replaceContent(param);
                    tempgen += ")";
                    
                    i = generation.indexOf(')',i);
                }
                
                //if the ckaracter is either a p.rule or a bracket, keep it
                else
                    tempgen += pArray[i];
            } //end current character
            
            generation = tempgen;
        } //end a derivation
        
        return generation;
    } //end generation of parametric string
    
    /**
     * method to return the parameters in parenthesis
     *
     * @param params    the parameters in a parenthesis
     */
    private String[] getParameters(String params) {
        params = params.substring(1,params.indexOf(')'));
        String[] tempParams = params.split(",");
        return tempParams;        
    }
    
    /**
     * method to replace the parameters in the parentheses 
     * of the string with given values
     * this method is capable of dealing with a string
     * containing more than one set of brackets
     * the content of each bracket is passed to replaceContent
     * method to replace the content of the brackets
     *
     * @param str   the parentheses with their content
     */
    private String replaceValuesInParenthesis(String str) {
        int i = 0;  //the current position in the string
        int j = 0;  //the position of the next opening bracket
        String content = new String();
        String newStr = new String();
        
        while (j!=-1) {

            newStr += str.substring(i,str.indexOf('(',i)+1);
            
            content = str.substring(str.indexOf('(',j)+1,str.indexOf(')',j));
            
            content = replaceContent(content);
            
            newStr += content;
                        
            i = str.indexOf(')', i+1);
            j = str.indexOf('(', i);
        }
        
        newStr += str.substring(i);
        return newStr;
    }
    
    /**
     * method to replace the value of the content in parentheses
     * three different possible content types are available
     * 1. one parameter required to replace it with its value
     * 2. two parameters that require multiplication
     * 3. two set of parameters separated by ','
     * case 1: the parameter is replaced from the appropriate 
     *         value from the parameters HashMap
     * case 2: the method multiplyItems is called to 
     *         replace and multiply the contents
     * case 3: the two sets are split and this method is called
     *         again for each parameter set separately
     *
     * @param str   content with no parentheses
     */
    private String replaceContent(String str) {
        String newStr = new String();
        String subContents[];
        if (str.indexOf(',')==-1 && str.indexOf('*')==-1) {
            if (parameters.containsKey(str))
                newStr = (String) parameters.get(str);
            else
                newStr = str;
        }
        else if (str.indexOf(',')==-1) {
            newStr = multiplyItems(str);
        }   
        else {
            subContents = str.split(",");
            newStr = replaceContent(subContents[0]);
            newStr += ",";
            newStr += replaceContent(subContents[1]);
        }
        
        return newStr;
    }
    
    /**
     * method to multiply the parameters in a parenthesis
     *
     * @param str   the parameters that need multiplication
     */
    private String multiplyItems(String str) {
        
        //get the two paramters of the multiplication
        String tempParam1 = str.substring(0,str.indexOf('*'));
        String tempParam2 = str.substring(str.indexOf('*')+1);
        String newStr = new String();
        
        float fParam1, fParam2, product;
        
        /**
         * replace the parameters with their value and convert it to 
         * a float number or simply convert the param to float if
         * not in parameters map
         */
        if (parameters.containsKey(tempParam1)) {
            tempParam1 = (String) parameters.get(tempParam1);
            fParam1 = Float.parseFloat(tempParam1);
        }
        else 
            fParam1 = Float.parseFloat(tempParam1);
        
        if (parameters.containsKey(tempParam2)) {
            tempParam2 = (String) parameters.get(tempParam2);
            fParam2 = Float.parseFloat(tempParam2);
        }
        else 
            fParam2 = Float.parseFloat(tempParam2);
        
        
        product = fParam1 * fParam2;
        
        newStr = "" + product;
        
        return newStr;
    }
    
    /**
     * method to print L-system details on the screen
     */
    public void printData() {
        System.out.println("The axiom of the system is: " + axiom);
        System.out.println("The number of derivations is: " + n);
        System.out.println("The production rules are: " + pRules.toString());
        System.out.println("The parametric values are: " + parameters.toString());
        System.out.println("The generation string is: " + generation);
    }
}