/**
 * InputWindow.java
 *
 * Created on 15 August 2004, 19:51
 *
 * This class creates the program's user interface and is responsible
 * for its functionality. The other classes assossiated with this program are:
 * TreeString, Turtle, Turtle3D, DrawTree, DrawTree3D
 *
 * @author  Alexander Dimitropoulos
 * @version 1
 */

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.border.*;

public class InputWindow extends JFrame{
    
    //parameters used for the creation of the GUI components
    private JTextField axiomArea, angleArea, derivArea, lengthArea, widthArea;
    private JTextArea productArea, outputArea;
    private JLabel axiomLabel, productLabel, angleLabel, derivLabel, lengthLabel, widthLabel, outputLabel;
    private JButton drawBtn, exitBtn, helpBtn;
    private JCheckBox Leaves, Colour, BackGround, Light, Texture, HiQuality;
    private JRadioButton TwoD, ThreeD, Cylinder, Polygon, Front, Side, Top, Rotation;
    private JComboBox FacesCombo, QualityCombo, TextureCombo;
    private String FacesSet[] = {"Faces: 4", "Faces: 6", "Faces: 8", "Faces: 12"};
    private String QualitySet[] = {"Low Shading","Hi Shading"};
    private String TextureStyle[] = {"Style 1", "Style 2", "Style 3", "Style 4"};
    private String texturePath[] = {"/Textures/treeTexture6.jpg","/Textures/treeTexture14.jpg","/Textures/treeTexture15.jpg","/Textures/treeTexture16.jpg"};    
    private int faces[] = {4,6,8,12};    
    private ButtonGroup modelButtons, viewButtons, shapeButtons;
    private Box buttonBox, optionsInputBox, textInputBox, settingsBox;
    private JPanel textInputPanel, buttonPanel, modelPanel,optionsPanel, viewsPanel, shapePanel, optimisationsPanel, texturePanel;
    private TitledBorder viewsBorder, shapeBorder, optimisationsBorder, textureBorder;
    
    //the interface's layout parameters
    private Container container;
    private GridBagLayout gbLayout;
    private GridBagConstraints gbConstraints;
    private UIManager.LookAndFeelInfo look[];
    
    //parameters used for input data
    private double angle;
    private float length, width;
    private int derivations;
    private String axiom;
    private String productions;
    private TreeString tree;
    private String tempAxiom, tempAngle, tempDeriv, tempLength, tempWidth;
    private String generatedString;
    private String output;
    
    /**
     * Creates a new instance of InputWindow
     */
    public InputWindow()
    {
        super("L-System Input");
        
        tree = new TreeString();
        generatedString = new String();
        
        //call of method to create the GUI components
        createGUI();
        
        //set the size of the GUI and display it
        setSize(450,450);
        show();
    }
    
    /**
     * method to create the GUI
     */
    private void createGUI() {
        
        container = getContentPane();
        container.setLayout(new BorderLayout(5,5));
        
        //axiom input field
        axiomArea = new JTextField(20);
        axiomLabel = new JLabel(" Axiom");
        axiomLabel.setToolTipText("eg: F");

        //productions input field
        productArea = new JTextArea(8,20);
        productLabel = new JLabel(" Production Rules");
        productLabel.setToolTipText("eg: F->FF-[-F+F+F]+[+F-F-F]\nf->f");
        
        //cangle input field
        angleArea = new JTextField(5);
        angleLabel = new JLabel(" Angle");
        angleLabel.setToolTipText("eg: 22.5");
        
        //derivations input field
        derivArea = new JTextField(5);
        derivLabel = new JLabel(" Derivations");
        derivLabel.setToolTipText("eg: 4");
        
        //length input field
        lengthArea = new JTextField(5);
        lengthLabel = new JLabel(" Length");
        lengthLabel.setToolTipText("Choose a number in the range 0-50");
        
        //width input field
        widthArea = new JTextField(5);
        widthLabel = new JLabel(" Width");
        widthLabel.setToolTipText("Choose a number (suggestion 1/10 of length)");
        
        //output window field
        outputArea = new JTextArea(5,20);
        outputLabel = new JLabel(" Output window");
        outputLabel.setToolTipText("This is a non-editable area where output from the system is given");
        outputArea.setEditable(false);
        outputArea.setToolTipText("In this area, information regarding the progress of the process is displayed");
        
        //2D options checkbox
        Leaves = new JCheckBox("Leaves");
        Leaves.setToolTipText("Choose if leaves are to be displayed");
        Colour = new JCheckBox("Colour");
        Colour.setToolTipText("If ticked, the colour of the model is brown and the leaves (if ticked) will be green");
        
        //optimisations checkboxes
        BackGround = new JCheckBox("Background");
        BackGround.setToolTipText("A default background is displayed");
        Light = new JCheckBox("Light");
        Light.setToolTipText("Light shines the object modelled");
        HiQuality = new JCheckBox("Hi-Quality");
        HiQuality.setToolTipText("The model is drawn with more detail");
        
        //texture checkbox
        Texture = new JCheckBox("Texture");
        
        //model type
        TwoD = new JRadioButton("2D",false);
        ThreeD = new JRadioButton("3D",true);
        modelButtons = new ButtonGroup();
        modelButtons.add(TwoD);
        modelButtons.add(ThreeD);
        
        //views selection buttons
        Front = new JRadioButton("Front",true);
        Top = new JRadioButton("Top",false);
        Side = new JRadioButton("Side",false);
        Rotation = new JRadioButton("Rotation",false);
        Rotation.setToolTipText("A constant rotation of the model along the vertical axis is displayed");
        viewButtons = new ButtonGroup();
        viewButtons.add(Front);
        viewButtons.add(Top);
        viewButtons.add(Side);
        viewButtons.add(Rotation);
        
        //shape selection buttons
        Cylinder = new JRadioButton("Cylinder",true);
        Cylinder.setToolTipText("The model is made of cylindrical sections");
        Polygon = new JRadioButton("Polygon",false);
        Polygon.setToolTipText("The model is made of polygonal sections");
        shapeButtons = new ButtonGroup();
        shapeButtons.add(Cylinder);
        shapeButtons.add(Polygon);
        
        //number of faces combo
        FacesCombo = new JComboBox(FacesSet);
        FacesCombo.setMaximumRowCount(4);
        FacesCombo.setToolTipText("Choose the number of faces that each polygon will have");
        
        //quality selection combo
        QualityCombo = new JComboBox(QualitySet);
        QualityCombo.setMaximumRowCount(4);
        QualityCombo.setToolTipText("Chose the quality of the shading of the objects");
                
        //style selection combo
        TextureCombo = new JComboBox(TextureStyle);
        TextureCombo.setMaximumRowCount(4);
        
        //introducing event handler
        InputWindowHandler handler = new InputWindowHandler();
        
        //creating process button
        drawBtn = new JButton("Generate Image");
        drawBtn.addActionListener(handler);
        
        //creating help button
        helpBtn = new JButton("Help");
        helpBtn.addActionListener(handler);
        
        //creating exit button
        exitBtn = new JButton("Exit");
        exitBtn.addActionListener(handler);
        
        
        //dividing the interface into 3 parts (boxes)
        textInputBox = Box.createVerticalBox();
        optionsInputBox = Box.createVerticalBox();
        buttonBox = Box.createHorizontalBox();
        
        //creating first box contents
        textInputPanel = new JPanel();
        gbLayout = new GridBagLayout();
        textInputPanel.setLayout(gbLayout);
        
        //instantiate gridbag constraints
        gbConstraints = new GridBagConstraints();
        gbConstraints.fill = GridBagConstraints.BOTH;
        
        //Adding all intput components
        addComponent(axiomLabel,0,0,2,1);
        addComponent(axiomArea,1,0,2,1);
        addComponent(productLabel,2,0,2,1);
        addComponent(new JScrollPane(productArea),3,0,2,8);
        addComponent(angleLabel,11,0,1,1);
        addComponent(angleArea,11,1,1,1);
        addComponent(derivLabel,12,0,1,1);
        addComponent(derivArea,12,1,1,1);
        addComponent(lengthLabel,13,0,1,1);
        addComponent(lengthArea,13,1,1,1);
        addComponent(widthLabel,14,0,1,1);
        addComponent(widthArea,14,1,1,1);
        addComponent(outputLabel,15,0,2,1);
        addComponent(new JScrollPane(outputArea),16,0,2,1);
        textInputBox.add(textInputPanel);
        
        //creating second box contents
        //Model option border
        modelPanel = new JPanel();
        modelPanel.setBorder(new TitledBorder("Model Type"));
        modelPanel.add(TwoD);
        modelPanel.add(ThreeD);
        optionsInputBox.add(modelPanel);
        
        //2D options border
        optionsPanel = new JPanel();
        optionsPanel.setBorder(new TitledBorder("2D Options"));
        optionsPanel.add(Leaves);
        optionsPanel.add(Colour);
        optionsInputBox.add(optionsPanel);
        
        //3D options border (Box containing 4 JPanels)
        settingsBox = Box.createVerticalBox();
        settingsBox.setBorder(new TitledBorder("3D Options"));
        
        //views options
        viewsPanel = new JPanel();
        viewsBorder = new TitledBorder("Views");
        viewsBorder.setTitleJustification(TitledBorder.CENTER);
        viewsPanel.setBorder(viewsBorder);
        viewsPanel.add(Front);
        viewsPanel.add(Side);
        viewsPanel.add(Top);
        viewsPanel.add(Rotation);
        settingsBox.add(viewsPanel);
        
        //shape options
        shapePanel = new JPanel();
        shapeBorder = new TitledBorder("Shape");
        shapeBorder.setTitleJustification(TitledBorder.CENTER);
        shapePanel.setBorder(shapeBorder);
        shapePanel.add(Cylinder);
        shapePanel.add(Polygon);
        shapePanel.add(FacesCombo);
        settingsBox.add(shapePanel);
        
        //optimisation options
        optimisationsPanel = new JPanel();
        optimisationsBorder = new TitledBorder("Optimisations");
        optimisationsBorder.setTitleJustification(TitledBorder.CENTER);
        optimisationsPanel.setBorder(optimisationsBorder);
        optimisationsPanel.add(Light);
        optimisationsPanel.add(HiQuality);
        optimisationsPanel.add(QualityCombo);
        settingsBox.add(optimisationsPanel);
        
        //texture options
        texturePanel = new JPanel();
        textureBorder = new TitledBorder("Texture");
        textureBorder.setTitleJustification(TitledBorder.CENTER);
        texturePanel.setBorder(textureBorder);
        texturePanel.add(BackGround);
        texturePanel.add(Texture);
        texturePanel.add(TextureCombo);
        settingsBox.add(texturePanel);
        
        //adding '3D options' box items 3rd from top on the rigth hand box
        optionsInputBox.add(settingsBox);
        
        //third box contents
        //adding the three buttons created previously to the corresponding box
        buttonPanel = new JPanel();
        buttonPanel.setLayout(new GridLayout(1,3));
        buttonPanel.add(drawBtn);
        buttonPanel.add(helpBtn);
        buttonPanel.add(exitBtn);
        buttonBox.add(buttonPanel);
        
        /**
         * add the three boxes created to the interface
         * the input part is located to the left of the interface
         * the options section is located on the rigth of the GUI
         * and the action buttons are on the bottom of the GUI
         */
        container.add(textInputBox, BorderLayout.WEST);
        container.add(optionsInputBox, BorderLayout.EAST);
        container.add(buttonBox, BorderLayout.SOUTH);
        
        //modifying the look of the UI to a 'Windows Style'
        look = UIManager.getInstalledLookAndFeels();
        setTheLookOfWindow(2);
    }
    
    /**
     * method to add a component to the GridBagLayout of textInputPanel
     *
     * @param c         the component to add
     * @param row       the row where the component is added
     * @param column    the column where the component is added
     * @param width     the width that teh component occupies
     * @param height    the height that the component occupies
     */
    private void addComponent( Component c, int row, int column, int width, int height )
    {
        // set gridx and gridy 
        gbConstraints.gridx = column;
        gbConstraints.gridy = row;

        // set gridwidth and gridheight
        gbConstraints.gridwidth = width;   
        gbConstraints.gridheight = height;

        // set constraints
        gbLayout.setConstraints( c, gbConstraints );  
        textInputPanel.add( c );
    }
    
    /**
     * method to set modify the look of the GUI
     *
     * @param i     the corresponding style of the window
     */
    private void setTheLookOfWindow(int i)
    {
        try {
            UIManager.setLookAndFeel(look[i].getClassName());
            SwingUtilities.updateComponentTreeUI(this);
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    
    /**
     * main window action listener
     */
    private class InputWindowHandler implements ActionListener {
        
        public void actionPerformed(ActionEvent e) {
            
            if (e.getActionCommand().equals("Generate Image")) {
                
                /**
                 * getting temp strings of axiom, angle, derivation and scale
                 * each string is trimmed to remove any unwanted spaces
                 * if any of these field is empty, show error message
                 */
                tempAxiom = axiomArea.getText().trim();
                tempAngle = angleArea.getText().trim();
                tempDeriv = derivArea.getText().trim();
                tempLength = lengthArea.getText().trim();
                tempWidth = widthArea.getText().trim();
                
                //case1: a non-parametric L-system is entered
                if (tempAxiom.indexOf('(')==-1) {
                    
                    /**
                     * if any of the Axiom, angle, derivations and length are not
                     * entered, an error message is returned
                     */
                    if (tempAxiom.equals(new String()) || 
                        tempAngle.equals(new String()) || 
                        tempDeriv.equals(new String()) || 
                        tempLength.equals(new String())) {
                            JOptionPane.showMessageDialog(null, 
                                "You have not entered all fields required!" , 
                                "Empty Field", JOptionPane.ERROR_MESSAGE);
                    }
                    
                    //width is checkes separately since it does not affect the 2D model
                    else if (ThreeD.isSelected() && tempWidth.equals(new String())) {
                        JOptionPane.showMessageDialog(null, 
                            "You have not entered all fields required!" , 
                            "Empty Field", JOptionPane.ERROR_MESSAGE);
                    }

                    //check if derivations number is not greater than zero
                    else if (Integer.parseInt(tempDeriv)<1) {
                        JOptionPane.showMessageDialog(null, 
                            "Derivations number should be a positive integer (>0)!",
                            "Negative Derivations Number",
                            JOptionPane.ERROR_MESSAGE);
                    }

                    else
                        generateImage();
                }
                
                //case2: a parametric L-system is entered
                else if (tempAxiom.indexOf('(')!=-1) {
                    
                    if (Integer.parseInt(tempDeriv)<1) {
                        JOptionPane.showMessageDialog(null, 
                            "Derivations number should be a positive integer (>0)!",
                            "Negative Derivations Number",
                            JOptionPane.ERROR_MESSAGE);
                    }
                    
                    else
                        generateImage();
                }
            }
            
            else if (e.getActionCommand().equals("Exit")) {
                System.exit(0);
            }
            
            else if (e.getActionCommand().equals("Help")) {
                displayHelp();               
            }
        
            
        }
            
    }
    
    /**
     * method to display a help window
     */
    private void displayHelp() {
        
        JTextArea helpText = new JTextArea(15,60);
        JScrollPane helpScroller = new JScrollPane(helpText,
            JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED,
            JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        
        String help = new String();
        
        help += " Axiom: The axiom of the L-System should be entered in \n" 
                + " this field.\ne.g.: F,F-F-F-F, !(5)F(30), A(10,1) etc.";
        help += "\n\n Production Rules: The production rules used to derive\n" +
                " the generation string should be entered here. Each rule\n" +
                " should be entered on a different line. Use '->' to " 
                + "\n represent the conversion. Use '=' for parameters" +
                "\ne.g.: F->F+F, wr=1.107";
        help += "\n\n Symbols that may be used:" +
                "\n F(l): draws a new section of length l." +
                "\n f(l): moves forward distance l without drawing." +
                "\n !(w): sets the width of the turtle to w" +
                "\n +(a): Turn left by angle a." +
                "\n -(a): Turn right by angle a." +
                "\n &(a): Pitch down by angle a." +
                "\n ^(a): Pitch up by angle a." +
                "\n <(a): Roll left by angle a." +
                "\n >(a): Roll right by angle a." +
                "\n |: Turn left by 180 degrees" +
                "\n $: Bring turtle to horizontal position" +
                "\n [: Start a branch." +
                "\n ]: Complete a branch.";
        help += "\n\n Angle: The angle increment should be entered in this " +
                "field.\n The angle should be in degrees and can be decimal." +
                "\n It is not required to enter angle in parametric rules" +
                "\ne.g.:22.5";
        help += "\n\n Derivations: The number of derivations should be " +
                "entered\n in this field. This should be an integer (>=1).";
        help += "\n\n Length: The length can be a decimal number and denotes " +
                "\n the length of the part to draw." +
                "\n It is not required to enter length in parametric rules";
        help += "\n\n Width: this decimal number is the width of the section " +
                "\n in 3D models only." +
                "\n It is not required to enter width in parametric rules";
        help += "\n\n Output Window: The output window returns information " +
                "\n about the progress of the execution";
        help += "\n\n\n -----------------------------------------------------" +
                "\n |                  Options Window                   |" +
                "\n -----------------------------------------------------";
        help += "\n\n Model Type: Selector of the type of model.";
        help += "\n\n 2D Options: Select if the model contains leaves or not." +
                "\n Also select if model has colour (brown for tree branches" +
                "\n and green for leaves, if selected)";
        help += "\n\n 3D Options: Views" +
                "\n There is a variety of views. The front view shows the " +
                "\n model from the front, the side view from the side " +
                "\n (after rotation through the vertical axis), the top " +
                "\n view shows the model from above and the rotation option" +
                "\n shows a model that rotates in real time";
        help += "\n\n 3D Options: Shape" +
                "\n Two different shapes may be used for the representation" +
                "\n of branches. The one is using cylinders and the other" +
                "\n using cylindrical triangular strip arrays";
        help += "\n\n 3D Options: Optimisations" +
                "\n Several options are available in order to optimise" +
                "\n the 3D model. Light can be made. Also," +
                "\n visible producing reflections on the model." +
                "\n the quality of the model can be increased, increasing" +
                "\n the number of polygons drawn for each section";
        help += "\n\n 3D Options: Texture" +
                "\n Texture can be added on the model by ticking the" +
                "\n appropriate box. The texture type can be selected" +
                "\n from a predefined list of five styles. A background" +
                "\n can also be added to the scene by selecting the" +
                "\n appropriate box";
        help += "\n\n\n -----------------------------------------------------" +
                "\n |               Different L-Systems                 |" +
                "\n -----------------------------------------------------";
        help += "\n\n 3D Tree 1:" +
                "\n Axiom: A(200,20)" +
                "\n Derivations: 10" +
                "\n Production rules: " +
                "\n A(l,w)->!(w)F(l)[&(a0)B(l*r2,w*wr)]>(d)A(l*r1,w*wr)" +
                "\n B(l,w)->!(w)F(l)[-(a2)$C(l*r2,w*wr)]C(l*r1,w*wr)" +
                "\n C(l,w)->!(w)F(l)[+(a2)$B(l*r2,w*wr)]B(l*r1,w*wr)" +
                "\n d=137.5" +
                "\n wr=0.707" +
                "\n r1=0.9" +
                "\n r2=0.6" +
                "\n a0=45" +
                "\n a2=45";
        help += "\n\n 3D Tree 2:" +
                "\n Axiom: A(200,20)" +
                "\n Derivations: 10" +
                "\n Production rules:" +
                "\n A(l,w)->!(w)F(l)[&(a1)B(l*r1,w*wr)]>(180)[&(a2)B(l*r2,w*wr)]" +
                "\n B(l,w)->!(w)F(l)[+(a1)$B(l*r1,w*wr)][-(a2)$B(l*r2,w*wr)]" +
                "\n wr=0.707" +
                "\n a1=5" +
                "\n a2=65" +
                "\n r1=0.9" +
                "\n r2=0.7";
        
        helpText.setText(help);
        helpText.setEditable(false);
        JOptionPane.showMessageDialog(null, helpScroller, "Help", JOptionPane.INFORMATION_MESSAGE);
    }
    
    /**
     * method to generate the image with the current inputs
     */
    private void generateImage() {
        
        output = new String();
        int tempDerivations = Integer.parseInt(derivArea.getText());
        
        //in case the string is not parametric, these fields are required
        if (axiomArea.getText().indexOf('(')==-1) {
            angle = Double.parseDouble(angleArea.getText());
            length = Float.parseFloat(lengthArea.getText());
            if (ThreeD.isSelected())
                width = Float.parseFloat(widthArea.getText());
        }
        
        
        String textureLocation = texturePath[TextureCombo.getSelectedIndex()];
        int qualitySetting = QualityCombo.getSelectedIndex()+1;
        int facesNo = faces[FacesCombo.getSelectedIndex()];
        
        if (!axiomArea.getText().trim().equals(axiom) || !productArea.getText().equals(productions) || tempDerivations!=derivations) {
            
            axiom = axiomArea.getText().trim();
            productions = productArea.getText();
            String tempProductions = productions;
            derivations = tempDerivations;
            
            tree.setAxiom(axiom);
            tree.setDerivations(derivations);

            String tempRule;
            tempProductions += "\n"; //adding \n (new line) at the end of the string to help next loop
            
            //for all the lines of the input rules window add the rules to the corresponding tree method
            while (tempProductions.indexOf('\n')>=0)
            {

                tempRule= tempProductions.substring(0,tempProductions.indexOf('\n')).trim();
                if (!tempRule.equals(new String()) && !tempRule.equals("\n")) {

                    if (tempRule.indexOf('>')!= -1)
                        tree.addRule(tempRule);
                    else if (tempRule.indexOf('=')!= -1)
                        tree.addParameter(tempRule);
                        
                }

                //remove from the string the production rule just added
                tempProductions = tempProductions.substring(tempProductions.indexOf('\n')+1, tempProductions.length());

            }
            
            //check if system is parametric or not to call the appropriate method
            if (axiom.indexOf('(')==-1) {
                output += "Generating string...\n";
                outputArea.setText(output);

            	generatedString = tree.generateString();
                
                output += "DONE \n\n";
                outputArea.setText(output);
            }
            else if (axiom.indexOf('(')!=-1) {
                output += "Generating string...\n";
                outputArea.setText(output);
                
            	generatedString = tree.generateParametricString();
                
                output += "DONE \n\n";
                outputArea.setText(output);
            }
        }

        //depending of the selection button 2D-3D, call the corresponding class
        if (ThreeD.isSelected() == true) {
            output += "Generating frame...\n";
            outputArea.setText(output);

            DrawTree3D image = new DrawTree3D(generatedString, angle, length, width, Top.isSelected(), Side.isSelected(), Rotation.isSelected(), Cylinder.isSelected(), Light.isSelected(), Texture.isSelected(), BackGround.isSelected(), HiQuality.isSelected(), facesNo, qualitySetting, textureLocation);
            
            output += "DONE";
            outputArea.setText(output);
        }
        else {
            output += "Generating frame...\n";
            outputArea.setText(output);
            
            DrawTree pic = new DrawTree(generatedString, angle, length, Leaves.isSelected(), Colour.isSelected());
            
            output += "DONE";
            outputArea.setText(output);
        }
        
    }
    
    /**
     * the main method to run the program
     */
    public static void main(String args[])
    { 
        InputWindow in = new InputWindow();
      
        in.addWindowListener(
            new WindowAdapter() {
                public void windowClosing(WindowEvent e) {
                    System.exit(0);
                }
            }
        );     
    }
}