/*
 * DrawTree3D.java
 *
 * Created on 02 September 2004, 16:37
 *
 * This class is responsible for the 3D modelling of DOL-Systems
 * and parametric L-Systems. THere are several options available
 * like lights, textures, different views, different objects
 *
 * @author  Alexander Dimitropoulos
 * @version 3
 */
import java.applet.Applet;
import java.awt.BorderLayout;
import javax.swing.JFrame;
import java.awt.event.*;
import java.awt.*;
import javax.swing.*;
import java.awt.GraphicsConfiguration;
import com.sun.j3d.utils.applet.MainFrame; 
import com.sun.j3d.utils.geometry.*;
import com.sun.j3d.utils.universe.*;
import com.sun.j3d.utils.behaviors.mouse.*;
import com.sun.j3d.utils.image.TextureLoader;
import javax.media.j3d.*;
import javax.vecmath.*;
import java.util.Stack;
import java.lang.Math.*;
import java.util.Enumeration;


public class DrawTree3D extends JFrame {
    
    private String treestr;         //the string to generate the tree
    private Turtle3D turtle;        //the turtle
    private Stack TurtleStack;      //the stack to hold turtle orientation
    private boolean treeRotation;
    private boolean treeTopView;
    private boolean treeSideView;
    private boolean cylindricalModel;
    private boolean light;
    private boolean backgrd;
    private boolean textureOn;
    private boolean qualityModel;
    int detail;                     //the detail denotes the number of faces of section
    private int coloringAttribute;
    private String filename;        //texture filename
    private SimpleUniverse simpleU;
    
    
    /** 
     * Creates a new instance of DrawTree3D
     */
    public DrawTree3D(String treestr, double angle, float length, float width, 
        boolean TopView, boolean SideView, boolean Rotation, boolean Cylinder, 
        boolean Light, boolean Texture, boolean Backgroung, boolean Quality, 
        int Detail, int Coloring, String textureFile) {
        
        super("3D Model");
        
        //set turtle position
        turtle = new Turtle3D(0.0, -0.5, 0.0);       //(X,Y,Z)
        TurtleStack = new Stack();
        
        //set turtle orientation Vertical Up (Heading vector +Y, Left vector -X, Up vector +Z)
        turtle.setHLU(new Vector3d(0,1,0), new Vector3d(-1,0,0), new Vector3d(0,0,1));
        
        this.treestr = new String(treestr);
        
        turtle.setAlpha(Math.toRadians(angle)); turtle.setLength(length); turtle.setWidth(width);
        treeRotation = Rotation;
        treeTopView = TopView;          //sets the view to Top
        treeSideView = SideView;
        cylindricalModel = Cylinder;    //sets the model to cylindrical or polygonal
        light = Light;                  //adds ligth to the scene
        backgrd = Backgroung;           //adds a background to the scene
        textureOn = Texture;            //adds texture to the model
        qualityModel = Quality;         //adds transition parts from one part to another (extra part on top of every section)
        detail = Detail;                //the detail denotes the number of faces of section
        coloringAttribute = Coloring;   //1 for fastest, 2 for nicest (visible on cylindrical (no texture) model only)
        filename = textureFile;
        
        Container c = getContentPane();
        
        GraphicsConfiguration config = SimpleUniverse.getPreferredConfiguration();

        Canvas3D canvas3D = new Canvas3D(config);
        
        
        c.setLayout(new BorderLayout());
        c.add(canvas3D, BorderLayout.CENTER);
        
        BranchGroup scene = create3DImage();
        
        // SimpleUniverse is a Convenience Utility class
        simpleU = new SimpleUniverse(canvas3D);

	// This will move the ViewPlatform back a bit so the
	// objects in the scene can be viewed.
        simpleU.getViewingPlatform().setNominalViewingTransform();
        
        // setLocalEyeViewing
        simpleU.getViewer().getView().setLocalEyeLightingEnable(true);
        
        simpleU.addBranchGraph(scene);
        
        this.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
        
        setSize(1024, 768);
        show();
    }
    
    /**
     *create 3D image
     *the image is constructed from 1 branch group and all objects are
     *added as childs of this branch group
     */
    public BranchGroup create3DImage() {
        
        BranchGroup objRoot = new BranchGroup();
        
        // Create Background
        objRoot.addChild(createBackground());
        
        //add light to the scene
        if (light) {
            
            Color3f whiteLight = new Color3f(1.0f,1.0f,1.0f);
            Vector3f direction = new Vector3f(-1.0f, -1.0f, -1.0f);
            DirectionalLight lightD1 = new DirectionalLight(whiteLight, direction);
            lightD1.setInfluencingBounds(new BoundingSphere());
            direction.normalize();
            
            objRoot.addChild(lightD1);
        }
        
        //Rotation and translation for the whole tree is added
        //Add mouse interaction
        TransformGroup objSpin = new TransformGroup();
        objSpin.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
        objSpin.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
        Alpha rotationAlpha = new Alpha(-1, 8000);
        RotationInterpolator rotator = new RotationInterpolator(rotationAlpha, objSpin);
        // a bounding sphere specifies a region a behavior is active
        // create a sphere centered at the origin with radius of 1
        rotator.setSchedulingBounds(new BoundingSphere());
        
        Transform3D rot = new Transform3D();
        
        if (treeTopView) {
            rot.rotX(Math.PI/2);
            objSpin = new TransformGroup(rot);
        }
        else if (treeSideView) {
            rot.rotY(Math.PI/2);
            objSpin = new TransformGroup(rot);
        }
        
        objRoot.addChild(objSpin);
        
        if (treeRotation)
            objSpin.addChild(rotator);
        
        //Mouse rotation interaction
        MouseRotate myMouseRotate = new MouseRotate();
        myMouseRotate.setTransformGroup(objSpin);
        myMouseRotate.setSchedulingBounds(new BoundingSphere());
        objRoot.addChild(myMouseRotate);

        MouseTranslate myMouseTranslate = new MouseTranslate();
        myMouseTranslate.setTransformGroup(objSpin);
        myMouseTranslate.setSchedulingBounds(new BoundingSphere());
        objRoot.addChild(myMouseTranslate);

        MouseZoom myMouseZoom = new MouseZoom();
        myMouseZoom.setTransformGroup(objSpin);
        myMouseZoom.setSchedulingBounds(new BoundingSphere());
        objRoot.addChild(myMouseZoom);
        
        
        //Image generation process
        char tree[] = treestr.toCharArray();
        char TreeChar;
        String tempNum = new String();
        float Num = 0;
        
        for (int i=0; i<tree.length; i++) {
            
            TreeChar = tree[i];
            
            if (tree.length!=(i+1) && tree[i+1]=='(') {
                tempNum = treestr.substring(treestr.indexOf('(',i)+1, treestr.indexOf(')',i));
                if (tempNum.indexOf(',')==-1)
                    Num = Float.parseFloat(tempNum);
            }
            
            if (TreeChar == 'F') {
                
                if (tree.length!=(i+1) && tree[i+1]=='(')
                    turtle.setLength(Num/1100);
                
                if (cylindricalModel) 
                    objSpin.addChild(createCylSection());
                else
                    objSpin.addChild(createPolySection());
                
                turtle.setNewPosition();
            }
            else if (TreeChar == 'f') {
                
                if (tree.length!=(i+1) && tree[i+1]=='(')
                    turtle.setLength(Num);
            
                turtle.setNewPosition();
            }
            else if (TreeChar == '-') {
                
                if (tree.length!=(i+1) && tree[i+1]=='(')
                    turtle.setAlpha(Math.toRadians(Num));
                
                turtle.turnLeftU();
            }
            else if (TreeChar == '+') {
                
                if (tree.length!=(i+1) && tree[i+1]=='(')
                    turtle.setAlpha(Math.toRadians(Num));
                
                turtle.turnRightU();
            }
            else if (TreeChar == '^') {
                
                if (tree.length!=(i+1) && tree[i+1]=='(')
                    turtle.setAlpha(Math.toRadians(Num));
                
                turtle.pitchDownL();
            }
            else if (TreeChar == '&') {
                
                if (tree.length!=(i+1) && tree[i+1]=='(')
                    turtle.setAlpha(Math.toRadians(Num));
                
                turtle.pitchUpL();
            }
            else if (TreeChar == '>') {
                
                if (tree.length!=(i+1) && tree[i+1]=='(')
                    turtle.setAlpha(Math.toRadians(Num));
                
                turtle.rollLeftH();
            }
            else if (TreeChar == '<') {
                
                if (tree.length!=(i+1) && tree[i+1]=='(')
                    turtle.setAlpha(Math.toRadians(Num));
                
                turtle.rollRightH();
            }
            else if (TreeChar == '|')
                turtle.turnAroundU();
            else if (TreeChar == '[')
                addTurtleToStack();
            else if (TreeChar == ']')
                popTurtleFromStack();
            else if (TreeChar == '!') {
                
                if (tree.length!=(i+1) && tree[i+1]=='(')
                    turtle.setWidth(Num/1100);
                else
                    turtle.setWidth(turtle.getWidth()*2/3);
            }
            else if (TreeChar == '$')
                turtle.rotateToVertical();
        }
        
        objRoot.compile();
        
        return objRoot;
    }
    
    /**
     *method to create a background of the scene
     */
    public Background createBackground() {
        Background bg;
        
        if (backgrd) {
            String imagefile = "/Textures/backGround.JPG";
            TextureLoader loader = new TextureLoader(imagefile, null);
            ImageComponent2D image = loader.getImage();
            bg = new Background(image);
        }
        else {
            bg = new Background(0.5f,0.6f,0.9f);
        }
        
        bg.setApplicationBounds(new BoundingSphere(new Point3d(), 100.0));
        
        return bg;
    }
    
    /**
     *this method creates the appearance of a geometric object
     *the triangula strip object is shown like a frame of its vertices
     */
    Appearance WireFrameAppearance() {

        Appearance materialAppear = new Appearance();
        PolygonAttributes polyAttrib = new PolygonAttributes();
        
        if (textureOn) {
            //texture
            polyAttrib.setCullFace(PolygonAttributes.CULL_NONE);
            materialAppear.setPolygonAttributes(polyAttrib);

            TexCoordGeneration tcg = new TexCoordGeneration(TexCoordGeneration.OBJECT_LINEAR,
                                                            TexCoordGeneration.TEXTURE_COORDINATE_2);
            materialAppear.setTexCoordGeneration(tcg);

            TextureLoader loader = new TextureLoader(filename, this);
            ImageComponent2D image = loader.getImage();
            Texture2D texture = new Texture2D(Texture.BASE_LEVEL, Texture.RGBA,
                                          image.getWidth(), image.getHeight());
            texture.setImage(0,image);
            texture.setEnable(true);
            materialAppear.setTexture(texture);
            //materialAppear.setTransparencyAttributes(new TransparencyAttributes(TransparencyAttributes.FASTEST, 0.1f));
        }
        else {
            //Wireframe appearance
            polyAttrib.setPolygonMode(PolygonAttributes.POLYGON_LINE);
            materialAppear.setPolygonAttributes(polyAttrib);

            ColoringAttributes redColoring = new ColoringAttributes();
            redColoring.setColor(0.0f, 1.0f, 0.0f);
            materialAppear.setColoringAttributes(redColoring);
        }
        
        return materialAppear;
    }
    
    Appearance CylinderAppearance() {
        Appearance appear = new Appearance();
        
        if (!textureOn) {
            // make modifications to default material properties
            Material material = new Material();
            material.setDiffuseColor(new Color3f(0.5f,0.4f,0.2f));
            //material.setSpecularColor(new Color3f(1.0f,1.0f,1.0f));
            material.setShininess(1000.0f);
            if (light)
                appear.setMaterial(material);

            //coloring of object
            ColoringAttributes colorAtt = new ColoringAttributes();
            colorAtt.setColor(0.5f,0.4f,0.2f);        
            if (coloringAttribute == 1)
                colorAtt.setShadeModel(ColoringAttributes.SHADE_FLAT);
            else
                colorAtt.setShadeModel(ColoringAttributes.SHADE_GOURAUD);

            appear.setColoringAttributes(colorAtt);
        }
        else {
            //Texture
            
            TextureLoader loader = new TextureLoader(filename, null);
            if (loader!=null) {
            ImageComponent2D image = loader.getImage();
            Texture2D texture = new Texture2D(Texture.BASE_LEVEL, Texture.RGBA, image.getWidth(), image.getHeight());
            texture.setImage(0,image);

            appear.setTexture(texture);
            }
        }
      return appear;
    }
    
    //polygon created parametrically (given centre and diameter)
    private BranchGroup createPolySection() {
        
        
        int triangleVertices = detail * 2 + 2;
        double[] coordinateData = new double[triangleVertices*3];
        double[] topCoordData = new double[triangleVertices*3];
        double X,Y,Z,A;
        X=0.0d; Y=0.0d; Z=0.0d;
	
        int i = 0;
        int j = 0;
        
        //the starting angle is a bit off 0 to prevent the side faces being parallel to the y-z plane
        //which has as a result this face being textured wrongly
        for (A=-Math.PI/8; A<2*Math.PI; A += Math.PI/(detail/2)) {
            
            coordinateData[i++]=X+turtle.getWidth()*Math.sin(A); coordinateData[i++]=turtle.getLength(); coordinateData[i++]=Z+turtle.getWidth()*Math.cos(A); //top
            if (turtle.getTempWidth() != 0) {
                coordinateData[i++]=X+turtle.getTempWidth()*Math.sin(A); coordinateData[i++]=Y; coordinateData[i++]=Z+turtle.getTempWidth()*Math.cos(A); //bottom
            }
            else {
                coordinateData[i++]=X+turtle.getWidth()*Math.sin(A); coordinateData[i++]=Y; coordinateData[i++]=Z+turtle.getWidth()*Math.cos(A); //bottom
            }
            
            //extra part on top of part
            if (qualityModel) {
                topCoordData[j++]=X+turtle.getWidth()*Math.sin(A) * 1/3; topCoordData[j++]=turtle.getLength()*7/6; topCoordData[j++]=Z+turtle.getWidth()*Math.cos(A) * 1/3; //top
                topCoordData[j++]=X+turtle.getWidth()*Math.sin(A); topCoordData[j++]=turtle.getLength(); topCoordData[j++]=Z+turtle.getWidth()*Math.cos(A); //bottom 
            }
        }
        
        turtle.setTempWidth(turtle.getWidth());
        
        
        int[] stripCount = {triangleVertices};
        GeometryInfo gi = new GeometryInfo(GeometryInfo.TRIANGLE_STRIP_ARRAY);
    
        gi.setCoordinates(coordinateData);
        gi.setStripCounts(stripCount);
     
        NormalGenerator ng = new NormalGenerator();
        ng.generateNormals(gi);
        gi.recomputeIndices();

        Shape3D part = new Shape3D();

        part.setAppearance(WireFrameAppearance());
        part.setGeometry(gi.getGeometryArray());
        
        BranchGroup hexFrame = new BranchGroup();
        
        //Rotate and Translate
        Transform3D rotate = new Transform3D();
        AxisAngle4d rotationAxisAngle = new AxisAngle4d();
        rotationAxisAngle = turtle.createRotationAxisAngle();
        rotate.setRotation(rotationAxisAngle);
        TransformGroup partRotate = new TransformGroup(rotate);
        
        Transform3D translate = new Transform3D();
        translate.set(new Vector3d(turtle.getX(),turtle.getY(),turtle.getZ()));
        TransformGroup partTranslate = new TransformGroup(translate);

        hexFrame.addChild(partTranslate);      //children addition of transform groups!
        partTranslate.addChild(partRotate);     //order of addition is important!!!
        partRotate.addChild(part);              //last element to add is the 3D shape  
        
        //top part for quality model
        if (qualityModel) {
            GeometryInfo giTop = new GeometryInfo(GeometryInfo.TRIANGLE_STRIP_ARRAY);
            giTop.setCoordinates(topCoordData);
            giTop.setStripCounts(stripCount);
            NormalGenerator ngTop = new NormalGenerator();
            ngTop.generateNormals(giTop);
            giTop.recomputeIndices();
            
            Shape3D topPart = new Shape3D();
            topPart.setAppearance(WireFrameAppearance());
            topPart.setGeometry(giTop.getGeometryArray());
            
            partRotate.addChild(topPart);
        }
        
        hexFrame.compile();
        
        return hexFrame;        
    }
    
    /**
     *method to create a new section
     *the new section is rotated and placed to the current turtle position
     */
    public BranchGroup createCylSection() {
        
        //create new branch group
        BranchGroup part = new BranchGroup();
        
        //move object to the position of the turtle
        Transform3D translate = new Transform3D();
        translate.set(new Vector3d(turtle.getX(),turtle.getY(),turtle.getZ()));
        TransformGroup moveToPlace = new TransformGroup(translate);
        
        part.addChild(moveToPlace);
        
        //rotate object to direction of turtle
        Transform3D rotate = new Transform3D();
        AxisAngle4d rotationAxisAngle = new AxisAngle4d();
        rotationAxisAngle = turtle.createRotationAxisAngle();
        rotate.setRotation(rotationAxisAngle);
        TransformGroup rotateToDirection = new TransformGroup(rotate);
        
        moveToPlace.addChild(rotateToDirection);
        
        //move object so that the origin is the centre of the bottom face
        Transform3D moveUp = new Transform3D();
        moveUp.set(new Vector3d(0,turtle.getLength()/2,0));
        TransformGroup moveOrigin = new TransformGroup(moveUp);
        
        rotateToDirection.addChild(moveOrigin);
        
        Cylinder partObj = new Cylinder(turtle.getWidth(), turtle.getLength(), Cylinder.GENERATE_NORMALS | Cylinder.GENERATE_TEXTURE_COORDS, CylinderAppearance());
        
        moveOrigin.addChild(partObj);
        
        if (qualityModel) {
            Transform3D moveSphereUp = new Transform3D();
            moveSphereUp.set(new Vector3d(0,turtle.getLength(),0));
            TransformGroup moveSphereOrigin = new TransformGroup(moveSphereUp);
            Sphere sphere = new Sphere(turtle.getWidth(), Sphere.GENERATE_NORMALS | Sphere.GENERATE_TEXTURE_COORDS, CylinderAppearance());
            rotateToDirection.addChild(moveSphereOrigin);
            moveSphereOrigin.addChild(sphere);
        }
        
        part.compile();
        
        return part;        
    }
    
    /**
     *method to add the current sate of the turtle to a stack 
     *so that it can be retrieved when ']' is found
     */
    public void addTurtleToStack() {
        Turtle3D tempTurtle3D = new Turtle3D(turtle.getX(), turtle.getY(), 
            turtle.getZ(), turtle.getAlpha(), turtle.getLength(), turtle.getWidth());
        tempTurtle3D.setHLU(turtle.getH(), turtle.getL(), turtle.getU());
        
        TurtleStack.push(tempTurtle3D);
    }
    
    /**
     *method to replace the current state of the turtle with
     *the top turtle of the stack
     */
    public void popTurtleFromStack() {
        turtle = (Turtle3D) TurtleStack.pop();
    }

}